import { useState, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, Clock, CheckCircle, MessageSquare, Star, FileText, Award } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyApplications.css'

const MyApplications = () => {
  const navigate = useNavigate()
  const [applications, setApplications] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('courseApplications')
    if (saved) {
      try {
        setApplications(JSON.parse(saved))
      } catch (e) {
        setApplications([])
      }
    } else {
      // Пример данных
      const defaultApps = [
        {
          id: 1,
          courseName: 'Основы алгоритмизации и программирования',
          startDate: '2025-05-15',
          status: 'active',
          review: null
        },
        {
          id: 2,
          courseName: 'Основы веб-дизайна',
          startDate: '2025-04-20',
          status: 'completed',
          review: {
            rating: 5,
            text: 'Отличный курс! Очень понравилась подача материала.'
          }
        },
        {
          id: 3,
          courseName: 'Основы проектирования баз данных',
          startDate: '2025-03-10',
          status: 'completed',
          review: null
        }
      ]
      setApplications(defaultApps)
      localStorage.setItem('courseApplications', JSON.stringify(defaultApps))
    }
  }, [])

  const handleReviewSubmit = (applicationId, rating, text) => {
    const updated = applications.map(app => {
      if (app.id === applicationId) {
        return {
          ...app,
          review: { rating, text }
        }
      }
      return app
    })
    setApplications(updated)
    localStorage.setItem('courseApplications', JSON.stringify(updated))
  }

  const handleComplete = (applicationId) => {
    const updated = applications.map(app => {
      if (app.id === applicationId) {
        return {
          ...app,
          status: 'completed'
        }
      }
      return app
    })
    setApplications(updated)
    localStorage.setItem('courseApplications', JSON.stringify(updated))
  }

  const formatDate = (dateString) => {
    const date = new Date(dateString)
    return date.toLocaleDateString('ru-RU', { 
      day: 'numeric', 
      month: 'long', 
      year: 'numeric' 
    })
  }

  return (
    <div className="applications-page-brutal">
      <div className="container">
        <section className="page-header-brutal">
          <h1 className="page-title-brutal">Мои заявки</h1>
          <p className="page-description-brutal">
            Просматривайте свои заявки, отслеживайте статус обучения и делитесь отзывами
          </p>
        </section>

        <ImageSlider />

        {applications.length === 0 ? (
          <div className="empty-state-brutal">
            <div className="empty-icon-wrapper-brutal">
              <FileText size={64} />
            </div>
            <h2 className="empty-title-brutal">Нет заявок</h2>
            <p className="empty-text-brutal">Создайте первую заявку на курс</p>
            <button 
              className="empty-button-brutal" 
              onClick={() => navigate('/enroll')}
            >
              Записаться на курс
            </button>
          </div>
        ) : (
          <section className="applications-list-brutal">
            <div className="applications-grid-brutal">
              {applications.map((application) => (
                <ApplicationCard
                  key={application.id}
                  application={application}
                  onReviewSubmit={handleReviewSubmit}
                  onComplete={handleComplete}
                  formatDate={formatDate}
                />
              ))}
            </div>
          </section>
        )}
      </div>
    </div>
  )
}

const ApplicationCard = ({ application, onReviewSubmit, onComplete, formatDate }) => {
  const [rating, setRating] = useState(application.review?.rating || 0)
  const [reviewText, setReviewText] = useState('')
  const [isSubmitting, setIsSubmitting] = useState(false)

  const handleSubmit = () => {
    if (rating === 0 || !reviewText.trim()) {
      alert('Пожалуйста, выберите рейтинг и напишите отзыв')
      return
    }
    setIsSubmitting(true)
    onReviewSubmit(application.id, rating, reviewText)
    setIsSubmitting(false)
    setReviewText('')
    setRating(0)
  }

  return (
    <div className={`application-item-brutal ${application.status}`}>
      <div className="item-header-brutal">
        <div className="item-number-brutal">#{application.id}</div>
        <div className={`status-indicator-brutal ${application.status}`}>
          {application.status === 'active' ? (
            <>
              <Clock size={16} />
              <span>Активна</span>
            </>
          ) : (
            <>
              <CheckCircle size={16} />
              <span>Завершена</span>
            </>
          )}
        </div>
      </div>
      <h3 className="item-title-brutal">{application.courseName}</h3>
      <div className="item-meta-brutal">
        <Calendar size={18} />
        <span>{formatDate(application.startDate)}</span>
      </div>
      
      {application.status === 'active' && (
        <div className="complete-section-brutal">
          <button
            className="complete-btn-brutal"
            onClick={() => onComplete(application.id)}
          >
            <CheckCircle size={20} />
            <span>Завершить курс</span>
          </button>
        </div>
      )}
      
      {application.status === 'completed' && (
        <div className="review-section-brutal">
          {application.review ? (
            <div className="review-display-brutal">
              <div className="review-stars-brutal">
                {Array.from({ length: 5 }).map((_, i) => (
                  <Star
                    key={i}
                    size={20}
                    fill={i < application.review.rating ? '#FFBE0B' : 'none'}
                    color={i < application.review.rating ? '#FFBE0B' : '#666'}
                  />
                ))}
              </div>
              <p className="review-text-brutal">{application.review.text}</p>
            </div>
          ) : (
            <div className="review-form-brutal">
              <div className="review-header-brutal">
                <MessageSquare size={20} />
                <span>Оставить отзыв о качестве образовательных услуг</span>
              </div>
              <p className="review-hint-brutal">Оцените курс по 5-балльной шкале и поделитесь своими впечатлениями</p>
              <div className="rating-brutal">
                <span className="rating-label-brutal">Рейтинг:</span>
                {Array.from({ length: 5 }).map((_, i) => (
                  <button
                    key={i}
                    type="button"
                    className="star-btn-brutal"
                    onClick={() => setRating(i + 1)}
                    title={`Оценить на ${i + 1} ${i === 0 ? 'звезду' : i < 4 ? 'звезды' : 'звезд'}`}
                  >
                    <Star
                      size={32}
                      fill={i < rating ? '#FFBE0B' : 'none'}
                      color={i < rating ? '#FFBE0B' : '#666'}
                    />
                  </button>
                ))}
                {rating > 0 && <span className="rating-value-brutal">{rating} из 5</span>}
              </div>
              <div className="textarea-wrapper-brutal">
                <label htmlFor={`review-text-${application.id}`} className="textarea-label-brutal">
                  Ваш отзыв:
                </label>
                <textarea
                  id={`review-text-${application.id}`}
                  className="review-textarea-brutal"
                  placeholder="Расскажите о вашем опыте обучения, что понравилось, что можно улучшить..."
                  rows="4"
                  value={reviewText}
                  onChange={(e) => setReviewText(e.target.value)}
                />
              </div>
              <button
                className="submit-review-brutal"
                onClick={handleSubmit}
                disabled={isSubmitting || rating === 0 || !reviewText.trim()}
              >
                {isSubmitting ? 'Отправка...' : 'Отправить отзыв'}
              </button>
            </div>
          )}
        </div>
      )}
    </div>
  )
}

export default MyApplications

